#!/usr/bin/perl

use warnings;
use strict;

use DOS::ConfigSys;
use POSIX qw< strftime >;
use File::Basename qw< basename >;
use Encode qw< encode decode >;

use constant DISKSIZE => 1150 * 1024;
#use constant DISKROOT => 'c:\dosgames';
use constant DISKROOT => 'a:';

{
  package DOS::Item;

  sub new { bless { @_[1..$#_] } => shift }

  sub cat   : lvalue { $_[0]->{cat}   }
  sub id    : lvalue { $_[0]->{id}    }
  sub title : lvalue { $_[0]->{title} }
  sub ram   : lvalue { $_[0]->{ram}   }
  sub disk  : lvalue { $_[0]->{disk}  }
  sub batch          { "gamebatches/"  . $_[0]->id . ".bat" }
  sub zip            { "zipfiles/" . $_[0]->id . ".zip" }
  sub zipsize        { (stat $_[0]->zip)[7] }
}

print STDERR "Reading index... ";
my (%gameid, @items, $count);
my $totsize = 0;
while(<>) {
  chomp;
  s/#.*//;
  next if /^\s*$/;
  my ($id, $title, $ram, $cat) = split /:/, $_;

  die "There are two games with the same id \"$id\" (conflicting game title: \"$title\")!\n"
    if $gameid{$id};
  $gameid{$id}++;

  my $item = DOS::Item->new(
    cat    => $cat,
    id     => $id,
    ram    => $ram,
    title  => $title,
  );

  die "Game \"$id\" does not have a batchfile!\n" unless -f $item->batch;
  die "Game \"$id\" does not have a zipfile!\n"   unless -f $item->zip;

  push @items, $item;
  $count++;
  $totsize += $item->zipsize;
}
printf STDERR "done, indexed %d games, %.2f KiB total.\n", $count, $totsize / 1024;

print STDERR "Grouping files to disks...        ";
my ($disknum, $cursize, @disk) = (0, 0, []);
my (%cat, %index, %cfgsys);

my @sorted = sort { $b->zipsize <=> $a->zipsize } @items;
while(my $item = shift @sorted) {
  printf STDERR "\b\b\b\b\b\b\b%03d/%03d", @items - @sorted, scalar @items;
  if($item->zipsize > DISKSIZE) {
    die "Game \"" . $item->id . "\" is too large to fit on one disk!\n";
  } elsif($cursize + $item->zipsize > DISKSIZE) {
    my $freespace = DISKSIZE - $cursize;
    if(defined(
      my $index = (grep { $sorted[$_]->zipsize <= $freespace } 0..$#sorted)[0]
    )) {
      push @sorted, splice(@sorted, 0, $index), $item;
      next;
    } else {
      @sorted = sort { $b->zipsize <=> $a->zipsize } @sorted;
      $disk[++$disknum] = [];
      $cursize = 0;
    }
  }

  $item->disk = $disknum;
  push @{ $disk[$disknum] }, $item;
  push @{ $index{uc substr $item->title, 0, 1} }, $item;
  push @{ $cat{$item->cat} }, $item;

  my $cfgitem = DOS::ConfigSys::Item->new(
    id    => sprintf("i%03d", DOS::ConfigSys::ID->new),
    title => sprintf("(%02d) %s", $item->disk + 1, $item->title),
    shell => sprintf('%s\command.com /k %s\test.bat %02d %s\batches\%s.bat %s',
      DISKROOT, DISKROOT, $item->disk + 1, DISKROOT, $item->id, DISKROOT),
    $item->ram ne "STD"
      ? (install_hook => DISKROOT . '\xmsdsk.exe ' . $item->ram . " e: /y")
      : (),
  );
  $cfgsys{$item} = $cfgitem;
  $cursize += $item->zipsize;
}
print STDERR "\b\b\b\b\b\b\bdone, using " . ($disknum + 1) . " disk(s):\n";
$totsize = 0;
for (my $i = 0; $i < @disk; $i++) {
  my $size = 0;
  $size += $_->zipsize for @{ $disk[$i] };
  $totsize += $size;
  printf STDERR "  #%02d: %3.2f KiB in %d files\n",
    $i + 1, $size / 1024, 0+@{ $disk[$i] };
}
printf STDERR "  Total: %.2f KiB in %d files\n", $totsize / 1024, 0+@items;

print STDERR "Generating config.sys... ";
my $cfgsys = DOS::ConfigSys->new;
my $bycat  = $cfgsys->new_menu(title => "Sortiert nach Kategorie");
my $bydisk = $cfgsys->new_menu(title => "Diskettenauswahl");
my $byname = $cfgsys->new_menu(title => "Index");
my $about  = $cfgsys->new_menu(title => "Über");

$about->new_text(title => "Sammlung von Ingo Blechschmidt <iblech\@web.de>");
$about->new_text(title => "Kompiliert am " . strftime "%c", localtime);
$about->new_text(title => "$count Spiele auf " . (0+@disk) . " Disketten");
$about->new_text(title => sprintf "%.2f Spiele/Diskette", $count/@disk);

foreach my $catname (sort keys %cat) {
  my $catmenu = $bycat->new_menu(title => $catname);
  foreach my $item (sort { uc $a->title cmp uc $b->title } @{ $cat{$catname} }) {
    push @{ $catmenu->items }, $cfgsys{$item};
  }
}

for(my $i = 0; $i < @disk; $i++) {
  my $diskmenu = $bydisk->new_menu(
    title => sprintf "Diskette (%02d) [%d Spiel%s]",
      $i + 1,
      0+@{ $disk[$i] },
      @{ $disk[$i] } > 1 ? "e" : "",
  );
  foreach my $item (sort { uc $a->title cmp uc $b->title } @{ $disk[$i] }) {
    push @{ $diskmenu->items }, $cfgsys{$item};
  }
}

foreach my $char (sort keys %index) {
  my $charmenu = $byname->new_menu(
    title => sprintf "Buchstabe %s [%d Spiel%s]",
      $char,
      0+@{ $index{$char} },
      @{ $index{$char} } > 1 ? "e" : "",
  );
  foreach my $item (sort { uc $a->title cmp $b->title } @{ $index{$char} }) {
    push @{ $charmenu->items }, $cfgsys{$item};
  }
}

print STDERR "done.\n";
(my $prolog = <<EOF) =~ s/ROOT/DISKROOT/eg;
[common]
device=ROOT\\himem.sys
install=ROOT\\xmsdsk.exe 768 e: /y
EOF
$cfgsys = encode "cp437", decode("utf8", $prolog . $cfgsys->as_text);
$cfgsys =~ s/\012/\015\012/g;

print STDERR "Copying files to disk directories...      ";
mkdir "gamedisks" or die "Couldn't mkdir \"gamedisks\": $!\n"
  unless -d "gamedisks";
for(my $i = 0; $i < @disk; $i++) {
  printf STDERR "\b\b\b\b\b%02d/%02d", $i + 1, 0+@disk;

  my $disksys = $cfgsys;
  my $cfgnum  = sprintf "(%02d)", $i + 1;
  $disksys =~ s/\Q$cfgnum\E/(XX)/g;

  my $dirnum = sprintf "%02d", $i + 1;
  mkdir "gamedisks/$dirnum" or die "Couldn't mkdir \"gamedisks/$dirnum\": $!\n"
    unless -d "gamedisks/$dirnum";

  open my $fh, ">", "gamedisks/$dirnum/config.sys" or
    die "Couldn't open \"gamedisks/$dirnum/config.sys\" for writing: $!\n";
  print $fh $disksys or
    die "Couldn't write to \"gamedisks/$dirnum/config.sys\": $!\n";
  close $fh or
    die "Couldn't close \"gamedisks/$dirnum/config.sys\": $!\n";

  open $fh, ">", "gamedisks/$dirnum/i-am-$dirnum" or
    die "Couldn't open \"gamedisks/$dirnum/i-am-$dirnum\" for writing: $!\n";
  close $fh or
    die "Couldn't close \"gamedisks/$dirnum/i-am-$dirnum\": $!\n";

  foreach my $dir (qw< batches zipfiles >) {
    mkdir "gamedisks/$dirnum/$dir" or die "Couldn't mkdir \"gamedisks/$dirnum/$dir\": $!\n"
      unless -d "gamedisks/$dirnum/$dir";
  }

  foreach my $item (@{ $disk[$i] }) {
    link $item->batch => "gamedisks/$dirnum/batches/" . $item->id . ".bat"
      or die "Couldn't link \"gamedisks/$dirnum/batches/" . $item->id . ".bat" . "\" to \"" . $item->batch . "\": $!\n";
    link $item->zip => "gamedisks/$dirnum/zipfiles/" . $item->id . ".zip"
      or die "Couldn't link \"gamedisks/$dirnum/zipfiles/" . $item->id . ".zip" . "\" to \"" . $item->zip . "\": $!\n";
  }
}
print STDERR "\b\b\b\b\bdone.\n";
