package LaTeX::Convert::MathML;

use warnings;
use strict;

use File::Temp  ();
use Cwd         ();
use Digest::MD5 ();
use File::Spec  ();

use constant CACHEDIR => ".cache";

our $mzlatex;
our $tempdir;

sub new {
  my $class = shift;

  $mzlatex = $class->test_mzlatex unless defined $mzlatex;
  return undef unless $mzlatex;
  return bless [], $class;
}

sub as_mathml {
  my ($self, $latex) = @_;

  my $md5    = Digest::MD5::md5_hex($latex);
  my $cached = File::Spec->catfile(CACHEDIR, "$md5.mathml");

  if(-f $cached) {
    mkdir CACHEDIR;
    open my $fh, "<", $cached or die "Couldn't open \"$cached\" for reading: $!\n";
      local $/;
      defined(my $mathml = <$fh>) or die "Couldn't read from \"$cached\": $!\n";
    close $fh or die "Couldn't close \"$cached\": $!\n";

    return $mathml;
  }

  $latex =~ s/\\mathrm/\\text/g;
  $latex =~ s/\\mathds/\\mathrm/g;

  $tempdir ||= File::Temp::tempdir(CLEANUP => 1);

  my ($fh, $name) = File::Temp::tempfile(DIR => $tempdir, SUFFIX => ".latex");
  print $fh <<LATEX;
\\documentclass{article}
\\usepackage{latexsym}
\\usepackage{amsmath}
\\usepackage{amssymb}
\\begin{document}

\\ensuremath{$latex}

\\end{document}
LATEX
  close $fh;

  my $old_cwd = Cwd::cwd;
  chdir $tempdir or die "Couldn't chdir() to \"$tempdir\": $!\n";
  $self->_nullstdout(sub {
    !system "mzlatex", $name
  })
		 or die "Couldn't run mzlatex successfully: $!\n";

  $name =~ s/latex$/xml/;
  open my $in, "<", $name or die "Couldn't open \"$name\" for reading: $!\n";
    local $/;
    <$in> =~ /<math[^>]+>(.*?)<\/math>/s;
    my $mathml = $1;
  close $in;

  chdir $old_cwd or die "Couldn't chdir to \"$old_cwd\": $!\n";

  open my $cfh, ">", $cached or die "Couldn't open \"$cached\" for writing: $!\n";
    print $cfh $mathml       or die "Couldn't write to \"$cached\": $!\n";
  close $cfh                 or die "Couldn't close \"$cached\": $!\n";

  return $mathml;
}

sub test_mzlatex {
  my $class = shift;

  $class->_nullstdout(sub { $class->_nullstderr(sub {
    local $?;
    system "tex4ht";
    if($? >= 0) {
      return 1;
    } else {
      return 0;
    }
  })});
}

sub _nullstdout {
  my ($self, $sub) = @_;

  open my $oldout, ">&", \*STDOUT or die "Can't dup STDOUT: $!\n";
  open STDOUT, ">", "/dev/null"   or die "Can't redirect STDOUT: $!\n";

  my $ret = &$sub;

  open STDOUT, ">&", $oldout      or die "Can't dup STDOUT: $!\n";

  return $ret;
}

sub _nullstderr {
  my ($self, $sub) = @_;

  open my $olderr, ">&", \*STDERR or die "Can't dup STDERR: $!\n";
  open STDERR, ">", "/dev/null"   or die "Can't redirect STDERR: $!\n";

  my $ret = &$sub;

  open STDERR, ">&", $olderr      or die "Can't dup STDERR: $!\n";

  return $ret;
}

1;
